import React, { Component } from 'react';
import cn from 'classnames';
import Select from '@material-ui/core/Select';
import FormControl from '@material-ui/core/FormControl';
import MenuItem from '@material-ui/core/MenuItem';
import Tooltip from '@material-ui/core/Tooltip';
import { withStyles } from '@material-ui/core/styles';
import { ipcRenderer, shell, remote } from 'electron';
import { ALL_LANGUAGES } from 'resources/strings/languages';
import { OS_IS_MAC, appVersion } from 'constants/versions';
import { MORE_INFO_URL } from 'constants/urls';
import logo from 'resources/images/logo_helmet.svg';
import help from 'resources/images/profile.svg';
import clsx from 'clsx';
import { History } from 'history';
import styles from './styles';
import * as Authentication from '../../library/Authentication';

type Props = {
    onLanguageSet: () => {},
    language: string,
    strings: Array,
    history: History,
    onSignOut: () => {},
    classes: {
        [key: string]: string
    }
};

class TitleBar extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            language: props.language,
            strings: props.strings,
            focused: true,
            maximized: false,
            // eslint-disable-next-line react/no-unused-state
            toolTipOpen: false
        };

        this.win = remote.getCurrentWindow();

        this.win.on('blur', () => {
            this.onWindowBlur();
        });
        this.win.on('focus', () => {
            this.onWindowFocus();
        });
    }

    static getDerivedStateFromProps(nextProps, prevState) {
        const { language } = prevState;
        if (language !== nextProps.language) {
            return {
                language: nextProps.language,
                strings: nextProps.strings
            };
        }
        return null;
    }

    onWindowBlur() {
        this.setState({ focused: false });
    }

    onWindowFocus() {
        this.setState({ focused: true });
    }

    openMoreInfo() {
        shell.openExternal(MORE_INFO_URL);
    }

    renderHelpTooltip() {
        const { strings } = this.state;
        const { classes, onSignOut } = this.props;
        const userDetails = Authentication.getUserDetails() || {};
        const email = Authentication.getUserEmail();

        return (
            <React.Fragment>
                <div className={classes.tooltipInnerContainer}>
                    <div className={classes.tooltipName}>
                        {userDetails.first_name} {userDetails.last_name}
                    </div>
                    <div className={classes.tooltipEmail}>{email}</div>

                    <div className={classes.tooltipSeparator} />

                    <div className={classes.tooltipAppName}>
                        <span>{strings.software_version}</span>
                        <span className={classes.tooltipAppVersion}>
                            - {appVersion}
                        </span>
                    </div>

                    <div className={classes.break} />
                    <div
                        onClick={() => this.openMoreInfo()}
                        onKeyPress={() => this.openMoreInfo()}
                        className={classes.tooltipMoreInfo}
                        role="button"
                    >
                        {strings.tooltip_more_info}
                    </div>

                    <div className={classes.tooltipSeparator} />

                    <div
                        className={classes.tooltipSignOut}
                        onClick={() => {
                            onSignOut();
                        }}
                    >
                        {strings.sign_out}
                    </div>
                </div>
            </React.Fragment>
        );
    }

    moveToScreen(path) {
        const { history } = this.props;

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(path);
        }, 100);
    }

    getMinimizeButtonClass() {
        const { maximized } = this.state;
        const { classes } = this.props;
        return cn(
            classes.trafficLight,
            maximized ? '' : classes.trafficLightMinimize
        );
    }

    renderWindowControls() {
        // Render window controls ("traffic light" buttons) only for MacOS
        if (!OS_IS_MAC) {
            return;
        }
        const { focused, maximized } = this.state;
        const { classes } = this.props;

        return (
            <div
                className={cn(
                    focused ? classes.focus : '',
                    classes.trafficLightsContainer
                )}
            >
                <div className={classes.trafficLights}>
                    <button
                        type="button"
                        className={cn(
                            classes.trafficLight,
                            classes.trafficLightClose
                        )}
                        onClick={() => ipcRenderer.send('close-window')}
                    />
                    <button
                        type="button"
                        className={this.getMinimizeButtonClass()}
                        onClick={() => {
                            if (maximized) return;
                            ipcRenderer.send('minimize-window');
                        }}
                    />
                    <button
                        type="button"
                        className={cn(
                            classes.trafficLight,
                            classes.trafficLightMaximizeDisabled
                        )}
                        onClick={() => this.setState({ maximized: false })}
                    />
                </div>
            </div>
        );
    }

    render() {
        const { language, onLanguageSet, classes } = this.props;
        const { strings } = this.state;
        const showSignInScreen =
            !Authentication.isSignedIn() ||
            Authentication.isWaitingForEmailVerification();
        return (
            <div
                className={clsx(
                    classes.container,
                    showSignInScreen ? classes.containerBare : null
                )}
            >
                {this.renderWindowControls()}
                {!showSignInScreen && (
                    <div className={classes.titleContainer}>
                        <img
                            className={classes.titleIcon}
                            src={logo}
                            alt="Logo"
                        />
                        <div className={classes.titleText}>
                            {strings.app_name}
                        </div>
                    </div>
                )}
                {!showSignInScreen && (
                    <div className={classes.rightSide}>
                        {Object.keys(ALL_LANGUAGES).length > 1 && (
                            <FormControl>
                                <Select
                                    value={language}
                                    onChange={event =>
                                        onLanguageSet(event.target.value)
                                    }
                                    MenuProps={{
                                        classes: {
                                            paper: classes.languagePaper
                                        }
                                    }}
                                    disableUnderline
                                    inputProps={{
                                        classes: {
                                            icon: classes.languageInputArrow,
                                            root: classes.languageInput,
                                            select: classes.languageSelect
                                        }
                                    }}
                                >
                                    {Object.keys(ALL_LANGUAGES).map(key => (
                                        <MenuItem
                                            className={cn(
                                                key === language
                                                    ? classes.selectedMenuItem
                                                    : ''
                                            )}
                                            key={key}
                                            value={key}
                                        >
                                            {key.toUpperCase()}
                                        </MenuItem>
                                    ))}
                                </Select>
                            </FormControl>
                        )}
                        <Tooltip
                            classes={{
                                tooltip: classes.tooltipContainer,
                                popperInteractive: classes.tooltipPoper
                            }}
                            interactive
                            onOpen={() => {
                                // eslint-disable-next-line react/no-unused-state
                                this.setState({ tooltipOpen: true });
                            }}
                            onClose={() => {
                                // eslint-disable-next-line react/no-unused-state
                                this.setState({ tooltipOpen: false });
                            }}
                            title={this.renderHelpTooltip()}
                        >
                            <img
                                className={classes.helpIcon}
                                src={help}
                                alt="Help"
                            />
                        </Tooltip>
                    </div>
                )}
            </div>
        );
    }
}

export default withStyles(styles)(TitleBar);
